# -*- coding: utf-8 -*-
"""Artificial_Immune_Networks-Pattern_Recognition.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1ZEJTyEXP3f8G2Hku30Ukp4zTP42i7g1V
"""

import numpy as np

class ArtificialImmuneNetwork:
    def __init__(self, num_antibodies, pattern_dim, affinity_threshold=0.8, mutation_rate=0.1, clone_factor=5):
        """
        Initializes the Artificial Immune Network.

        Args:
            num_antibodies (int): Initial number of random antibodies.
            pattern_dim (int): Dimensionality of the patterns (e.g., number of features).
            affinity_threshold (float): Minimum affinity for an antibody to be considered a match.
            mutation_rate (float): Probability of mutation for a gene in a cloned antibody.
            clone_factor (int): How many clones to generate for a selected antibody.
        """
        self.num_antibodies = num_antibodies
        self.pattern_dim = pattern_dim
        self.affinity_threshold = affinity_threshold
        self.mutation_rate = mutation_rate
        self.clone_factor = clone_factor

        # Antibodies are represented as binary vectors (for simplicity in this example)
        # In more complex scenarios, they could be real-valued, or more abstract representations.
        self.antibodies = np.random.randint(0, 2, size=(num_antibodies, pattern_dim))
        self.memory_cells = [] # To store highly effective antibodies

    def _calculate_affinity(self, antibody, antigen):
        """
        Calculates the affinity between an antibody and an antigen.
        Here, it's a simple Hamming distance-like measure (proportion of matching bits).
        Higher value means higher affinity.
        """
        return np.sum(antibody == antigen) / self.pattern_dim

    def _mutate(self, antibody):
        """
        Mutates an antibody by flipping some bits based on the mutation rate.
        """
        mutated_antibody = np.copy(antibody)
        for i in range(self.pattern_dim):
            if np.random.rand() < self.mutation_rate:
                mutated_antibody[i] = 1 - mutated_antibody[i]  # Flip the bit
        return mutated_antibody

    def train(self, antigens, generations=10):
        """
        Trains the AIN using a set of antigens.
        Implements a simplified clonal selection principle.

        Args:
            antigens (list of np.array): A list of patterns (antigens) to learn.
            generations (int): Number of training iterations.
        """
        print("Starting AIN training...")
        for gen in range(generations):
            print(f"\n--- Generation {gen+1}/{generations} ---")
            new_antibodies_to_add = []

            for antigen in antigens:
                # 1. Antigen Presentation and Recognition
                affinities = [self._calculate_affinity(ab, antigen) for ab in self.antibodies]
                best_antibody_idx = np.argmax(affinities)
                best_affinity = affinities[best_antibody_idx]
                best_antibody = self.antibodies[best_antibody_idx]

                print(f"  Antigen: {antigen}")
                print(f"  Best matching antibody (initial pool): {best_antibody} (Affinity: {best_affinity:.2f})")

                # 2. Clonal Selection and Proliferation (if affinity is high enough)
                if best_affinity >= self.affinity_threshold:
                    print(f"  Affinity above threshold. Cloning antibody...")
                    for _ in range(self.clone_factor):
                        cloned_antibody = np.copy(best_antibody)
                        mutated_cloned_antibody = self._mutate(cloned_antibody)
                        new_antibodies_to_add.append(mutated_cloned_antibody)
                else:
                    print(f"  Affinity below threshold. No cloning for this antigen.")

            # Add newly generated antibodies to the main pool
            if new_antibodies_to_add:
                # Simple diversity maintenance: add new antibodies and remove some old ones randomly
                # In a real AIN, suppression/idiotypic network interactions would be more complex.
                self.antibodies = np.vstack((self.antibodies, np.array(new_antibodies_to_add)))
                np.random.shuffle(self.antibodies)
                self.antibodies = self.antibodies[:self.num_antibodies] # Keep a fixed size by truncation
                print(f"  Antibody pool size after generation: {len(self.antibodies)}")

            # Update memory cells (simplified: keep the very best from current pool)
            for ab in self.antibodies:
                is_unique = True
                for mem_ab in self.memory_cells:
                    if np.array_equal(ab, mem_ab):
                        is_unique = False
                        break
                if is_unique and self._calculate_affinity(ab, antigens[0]) > 0.9: # Example threshold for memory
                    self.memory_cells.append(ab)
                    print(f"  Added antibody to memory: {ab}")

        print("\nTraining complete.")
        print(f"Final antibody pool size: {len(self.antibodies)}")
        print(f"Memory cells: {len(self.memory_cells)}")
        for mem_ab in self.memory_cells:
            print(f"  Memory cell: {mem_ab}")

    def recognize(self, pattern):
        """
        Recognizes a new pattern by checking affinity with existing antibodies (including memory cells).

        Args:
            pattern (np.array): The pattern to recognize.

        Returns:
            bool: True if the pattern is recognized, False otherwise.
            np.array: The best matching antibody.
            float: The highest affinity found.
        """
        all_detectors = list(self.antibodies) + list(self.memory_cells)
        if not all_detectors:
            print("No detectors available to recognize patterns.")
            return False, None, 0.0

        affinities = [self._calculate_affinity(ab, pattern) for ab in all_detectors]
        best_match_idx = np.argmax(affinities)
        best_match_antibody = all_detectors[best_match_idx]
        highest_affinity = affinities[best_match_idx]

        print(f"\n--- Recognizing pattern: {pattern} ---")
        print(f"  Best matching detector: {best_match_antibody} (Affinity: {highest_affinity:.2f})")

        if highest_affinity >= self.affinity_threshold:
            print("  Pattern recognized!")
            return True, best_match_antibody, highest_affinity
        else:
            print("  Pattern not recognized (affinity below threshold).")
            return False, best_match_antibody, highest_affinity

# --- Example Usage ---
if __name__ == "__main__":
    # Define some binary patterns (antigens) for training
    # Let's say we want to recognize patterns similar to [1,1,0,0] and [0,0,1,1]
    pattern_dimension = 4
    training_antigens = [
        np.array([1, 1, 0, 0]),
        np.array([1, 0, 0, 0]), # A variation of the first
        np.array([0, 0, 1, 1]),
        np.array([0, 1, 1, 1])  # A variation of the second
    ]

    # Initialize the AIN
    ain = ArtificialImmuneNetwork(
        num_antibodies=20,          # Start with 20 random antibodies
        pattern_dim=pattern_dimension,
        affinity_threshold=0.75,    # High affinity needed for recognition
        mutation_rate=0.05,         # Small chance of mutation
        clone_factor=3              # Generate 3 clones per selected antibody
    )

    # Train the AIN
    ain.train(training_antigens, generations=5)

    # Test with new patterns
    print("\n--- Testing Recognition ---")

    test_pattern_1 = np.array([1, 1, 0, 1]) # Similar to [1,1,0,0]
    recognized, best_ab, affinity = ain.recognize(test_pattern_1)

    test_pattern_2 = np.array([0, 0, 0, 0]) # Not similar to trained patterns
    recognized, best_ab, affinity = ain.recognize(test_pattern_2)

    test_pattern_3 = np.array([0, 1, 1, 0]) # Similar to [0,1,1,1]
    recognized, best_ab, affinity = ain.recognize(test_pattern_3)

    test_pattern_4 = np.array([1, 1, 1, 1]) # Another pattern
    recognized, best_ab, affinity = ain.recognize(test_pattern_4)