# -*- coding: utf-8 -*-
"""Artificial_Immuse_Networks-Optimization.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/11bQ1XRKaDeegY1WJbK2vJewh7-A1L3yZ
"""

import numpy as np
import matplotlib.pyplot as plt

class ClonalSelectionOptimizer:
    def __init__(self,
                 objective_function,
                 search_range,
                 num_antibodies,
                 num_clones_per_antibody,
                 hypermutation_rate,
                 diversity_rate,
                 max_generations):
        """
        Initializes the Clonal Selection Optimization algorithm.

        Args:
            objective_function (callable): The function to be minimized.
                                         It should take a single float (x) and return a float (f(x)).
            search_range (tuple): A tuple (min_val, max_val) defining the search space for x.
            num_antibodies (int): The number of antibodies (solution candidates) in the population.
            num_clones_per_antibody (int): How many clones to generate from each selected antibody.
            hypermutation_rate (float): Controls the intensity of mutation (e.g., standard deviation for Gaussian noise).
            diversity_rate (float): Proportion of the population to be replaced by random antibodies
                                    to maintain diversity.
            max_generations (int): Maximum number of iterations for the optimization.
        """
        self.objective_function = objective_function
        self.search_range = search_range
        self.num_antibodies = num_antibodies
        self.num_clones_per_antibody = num_clones_per_antibody
        self.hypermutation_rate = hypermutation_rate
        self.diversity_rate = diversity_rate
        self.max_generations = max_generations

        # Antibodies are represented as single float values (x in f(x))
        # Initialize antibodies randomly within the search range
        self.antibodies = np.random.uniform(search_range[0], search_range[1], num_antibodies)
        self.best_solution_history = []
        self.best_affinity_history = []

    def _calculate_affinity(self, antibody_value):
        """
        Calculates the affinity (fitness) of an antibody.
        For minimization problems, lower function value means higher affinity.
        We can use 1 / (1 + f(x)) or similar for a positive, bounded affinity.
        Or, just use -f(x) directly for maximization, or f(x) and sort for minimization.
        Here, we will directly use f(x) and always select the minimum.
        """
        return self.objective_function(antibody_value)

    def _mutate(self, antibody_value, affinity):
        """
        Mutates an antibody. The mutation rate (intensity) can be inversely proportional to affinity
        (i.e., less fit antibodies get more mutation to explore, or more fit antibodies get smaller mutations
        for fine-tuning, or a fixed rate).
        Here, we use a fixed hypermutation rate, adding Gaussian noise.
        """
        # The mutation strength can be inversely proportional to affinity (fitness)
        # to encourage fine-tuning for good solutions and broader search for poor ones.
        # For simplicity, we use a fixed mutation rate here.
        mutation_strength = self.hypermutation_rate
        mutated_value = antibody_value + np.random.normal(0, mutation_strength)

        # Ensure the mutated value stays within the search range
        mutated_value = np.clip(mutated_value, self.search_range[0], self.search_range[1])
        return mutated_value

    def optimize(self):
        """
        Runs the Clonal Selection Optimization algorithm.
        """
        print("Starting Clonal Selection Optimization...")

        for generation in range(self.max_generations):
            # Evaluate all antibodies
            affinities = [self._calculate_affinity(ab) for ab in self.antibodies]
            sorted_indices = np.argsort(affinities) # For minimization, sort ascending

            # Select the best antibodies (top N, N being self.num_antibodies for cloning)
            # In some CSAs, only a subset of the best are chosen for cloning.
            # Here, we will clone all antibodies, but higher affinity ones will produce better clones.
            selected_antibodies = self.antibodies[sorted_indices]
            selected_affinities = np.array(affinities)[sorted_indices]

            new_population = []
            for i in range(self.num_antibodies):
                parent_antibody = selected_antibodies[i]
                parent_affinity = selected_affinities[i]

                # Generate clones
                clones = []
                for _ in range(self.num_clones_per_antibody):
                    # Higher affinity (lower f(x)) means less mutation usually,
                    # but here we'll use a fixed rate for simplicity.
                    # A common variant: mutation rate is inversely proportional to affinity
                    # (1 - normalized_affinity) * base_mutation_rate
                    mutated_clone = self._mutate(parent_antibody, parent_affinity)
                    clones.append(mutated_clone)

                # Evaluate clones
                clone_affinities = [self._calculate_affinity(c) for c in clones]

                # Select the best clone (and potentially the parent) to survive
                # In some CSAs, the best clone replaces the parent if it's better.
                # Here, we just add the best clone to the new population.
                best_clone_idx = np.argmin(clone_affinities) # Best clone has lowest f(x)
                best_clone = clones[best_clone_idx]

                # Add the best clone (and potentially the parent if it's better)
                if parent_affinity < clone_affinities[best_clone_idx]: # If parent is better than its best clone
                    new_population.append(parent_antibody)
                else:
                    new_population.append(best_clone)

            # Introduce diversity (randomly generated antibodies)
            num_to_replace = int(self.num_antibodies * self.diversity_rate)
            random_antibodies = np.random.uniform(self.search_range[0], self.search_range[1], num_to_replace)

            # Combine old (selected and cloned) population with new random antibodies
            # Ensure the population size remains constant
            self.antibodies = np.array(new_population[:self.num_antibodies - num_to_replace] + random_antibodies.tolist())
            np.random.shuffle(self.antibodies) # Shuffle to mix

            # Find and store the best solution in the current generation
            current_best_x = self.antibodies[np.argmin([self._calculate_affinity(ab) for ab in self.antibodies])]
            current_best_f_x = self._calculate_affinity(current_best_x)

            self.best_solution_history.append(current_best_x)
            self.best_affinity_history.append(current_best_f_x)

            print(f"Generation {generation+1}/{self.max_generations}: "
                  f"Best x = {current_best_x:.4f}, f(x) = {current_best_f_x:.4f}")

        final_best_x = self.best_solution_history[-1]
        final_best_f_x = self.best_affinity_history[-1]
        print(f"\nOptimization complete.")
        print(f"Optimal x found: {final_best_x:.4f}")
        print(f"Minimum f(x) found: {final_best_f_x:.4f}")
        return final_best_x, final_best_f_x

    def plot_results(self):
        """Plots the objective function and the optimization history."""
        x_vals = np.linspace(self.search_range[0], self.search_range[1], 500)
        y_vals = [self.objective_function(x) for x in x_vals]

        plt.figure(figsize=(10, 6))
        plt.plot(x_vals, y_vals, label='Objective Function $f(x) = x^2 + 5\sin(x)$', color='blue')
        plt.scatter(self.best_solution_history, self.best_affinity_history,
                    color='red', marker='o', label='Best Solution per Generation')
        plt.scatter(self.best_solution_history[-1], self.best_affinity_history[-1],
                    color='green', marker='X', s=200, label='Final Best Solution')
        plt.title('Clonal Selection Optimization Progress')
        plt.xlabel('x')
        plt.ylabel('f(x)')
        plt.legend()
        plt.grid(True)
        plt.show()

        plt.figure(figsize=(10, 4))
        plt.plot(range(1, self.max_generations + 1), self.best_affinity_history, color='purple')
        plt.title('Best f(x) Value Over Generations')
        plt.xlabel('Generation')
        plt.ylabel('f(x)')
        plt.grid(True)
        plt.show()


# --- Example Usage ---
if __name__ == "__main__":
    # Define the objective function to minimize
    def objective_function(x):
        return x**2 + 5 * np.sin(x)

    # Define the search range for x
    search_range = (-5, 5) # Let's search between -5 and 5

    # Create and run the optimizer
    cso = ClonalSelectionOptimizer(
        objective_function=objective_function,
        search_range=search_range,
        num_antibodies=50,             # Number of solution candidates
        num_clones_per_antibody=10,    # How many variants generated from each good solution
        hypermutation_rate=0.2,        # Standard deviation of mutation noise
        diversity_rate=0.1,            # 10% of population replaced by random ones each generation
        max_generations=100            # Number of optimization iterations
    )

    best_x, min_fx = cso.optimize()

    # Plot the results
    cso.plot_results()

    # Compare with known global minimum (if easily calculable or known from literature)
    # For x^2 + 5sin(x), the global minimum is approximately at x = -1.114 with f(x) = -3.768
    print(f"\nKnown approximate global minimum: x = -1.114, f(x) = -3.768")