# -*- coding: utf-8 -*-
"""Clonal_Selection_Algorithm.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1T-u5sA3wkY0a9uSEKO5u8TrEs95Atmsk
"""

import numpy as np
import matplotlib.pyplot as plt

# --- 1. Problem Definition ---
def objective_function(x):
    """
    The objective function to minimize.
    Here, a simple parabola: (x-5)^2. The minimum is at x=5, f(x)=0.
    """
    return (x - 5)**2

def affinity_function(value):
    """
    Converts objective function value to affinity.
    For minimization, lower objective value means higher affinity.
    We'll use 1 / (1 + value) to map to a [0, 1] range, where 1 is best.
    """
    return 1 / (1 + value)

# --- 2. CSA Parameters ---
POPULATION_SIZE = 50       # Number of antibodies
NUM_GENERATIONS = 100      # Number of iterations
NUM_CLONES_PER_ANTIBODY = 5 # Number of clones generated for each selected antibody
SELECTION_PERCENTAGE = 0.2  # Percentage of top antibodies selected for cloning (e.g., 20%)
RANDOM_REPLACEMENT_PERCENTAGE = 0.1 # Percentage of lowest-affinity antibodies replaced by random ones

# Mutation parameters
MUTATION_STRENGTH_MAX = 0.5 # Maximum mutation step size
MUTATION_STRENGTH_MIN = 0.01 # Minimum mutation step size (for high affinity)

# --- 3. Antibody Representation ---
# An antibody is simply a float (x-value) in this 1D problem.

# --- 4. Clonal Selection Algorithm ---

def clonal_selection_algorithm():
    # 1. Initialize Population
    # Antibodies are randomly initialized within a search space (e.g., [0, 10])
    population = np.random.uniform(0, 10, POPULATION_SIZE)

    best_solution_history = []
    best_affinity_history = []

    # Initialize memory cells (to store the overall best solution found)
    global_best_antibody = None
    global_best_affinity = -np.inf # Start with negative infinity for maximization

    for generation in range(NUM_GENERATIONS):
        # 2. Evaluate Population Affinity
        affinities = np.array([affinity_function(objective_function(x)) for x in population])

        # Sort population and affinities by affinity in descending order
        sorted_indices = np.argsort(affinities)[::-1]
        population_sorted = population[sorted_indices]
        affinities_sorted = affinities[sorted_indices]

        # Update global best solution
        if affinities_sorted[0] > global_best_affinity:
            global_best_affinity = affinities_sorted[0]
            global_best_antibody = population_sorted[0]

        best_solution_history.append(global_best_antibody)
        best_affinity_history.append(global_best_affinity)

        print(f"Generation {generation+1}: Best x = {global_best_antibody:.4f}, Objective = {objective_function(global_best_antibody):.4f}")

        # 3. Select Antibodies for Cloning
        num_selected = int(POPULATION_SIZE * SELECTION_PERCENTAGE)
        selected_antibodies = population_sorted[:num_selected]
        selected_affinities = affinities_sorted[:num_selected]

        clones = []
        # 4. Cloning and 5. Somatic Hypermutation
        for i, antibody in enumerate(selected_antibodies):
            affinity = selected_affinities[i]

            # Calculate mutation strength: inversely proportional to affinity
            # High affinity -> small mutation (fine-tuning)
            # Low affinity -> large mutation (exploration)
            # We normalize affinity to [0, 1] for this calculation.
            normalized_affinity = (affinity - affinities.min()) / (affinities.max() - affinities.min() + 1e-9) # Avoid div by zero
            mutation_strength = MUTATION_STRENGTH_MAX - (normalized_affinity * (MUTATION_STRENGTH_MAX - MUTATION_STRENGTH_MIN))

            for _ in range(NUM_CLONES_PER_ANTIBODY):
                # Apply Gaussian mutation
                mutated_clone = antibody + np.random.normal(0, mutation_strength)

                # Ensure clones stay within search bounds (optional, but good practice)
                mutated_clone = np.clip(mutated_clone, 0, 10)
                clones.append(mutated_clone)

        # 6. Re-evaluate Clones
        clones_affinities = np.array([affinity_function(objective_function(x)) for x in clones])

        # Combine selected parents and their clones for next selection
        combined_population = np.concatenate((selected_antibodies, np.array(clones)))
        combined_affinities = np.concatenate((selected_affinities, clones_affinities))

        # 7. Clonal Selection (Environmental Selection)
        # Select the best 'POPULATION_SIZE - num_random_replacements' individuals
        # from the combined pool to form the next generation's "core"

        # Sort combined population by affinity
        combined_sorted_indices = np.argsort(combined_affinities)[::-1]
        combined_population_sorted = combined_population[combined_sorted_indices]

        # New population starts with the best from the combined pool
        new_population = combined_population_sorted[:POPULATION_SIZE] # Take enough to fill up the pop, will be filled with random later

        # 9. Diversity Introduction (Replacement of low-affinity individuals)
        num_random_replacements = int(POPULATION_SIZE * RANDOM_REPLACEMENT_PERCENTAGE)
        if num_random_replacements > 0:
            random_antibodies = np.random.uniform(0, 10, num_random_replacements)
            # Replace the lowest 'num_random_replacements' individuals in new_population
            # Or, more cleanly, ensure the new_population size is filled up this way
            # Here, we'll re-select POPULATION_SIZE - num_random_replacements
            # and add random ones to make up the size

            new_population = combined_population_sorted[:(POPULATION_SIZE - num_random_replacements)]
            new_population = np.concatenate((new_population, random_antibodies))

            # If the population size isn't exactly matched by the above, trim/pad
            if len(new_population) > POPULATION_SIZE:
                new_population = new_population[:POPULATION_SIZE]
            elif len(new_population) < POPULATION_SIZE:
                new_population = np.concatenate((new_population, np.random.uniform(0, 10, POPULATION_SIZE - len(new_population))))


        population = new_population # The new population for the next generation


    # After all generations, return the overall best found
    return global_best_antibody, objective_function(global_best_antibody), best_solution_history, best_affinity_history

# --- Run the Algorithm ---
if __name__ == "__main__":
    best_x, best_obj, history_x, history_affinity = clonal_selection_algorithm()

    print("\n--- Optimization Results ---")
    print(f"Optimal x found: {best_x:.4f}")
    print(f"Minimum objective value: {best_obj:.4f}")

    # --- Visualization ---
    generations = np.arange(NUM_GENERATIONS)

    # Plot objective function
    x_values = np.linspace(0, 10, 400)
    y_values = objective_function(x_values)

    plt.figure(figsize=(12, 6))

    plt.subplot(1, 2, 1)
    plt.plot(x_values, y_values, label='Objective Function $f(x) = (x-5)^2$', color='blue')
    plt.axvline(x=5, color='red', linestyle='--', label='True Minimum at x=5')
    plt.scatter(history_x, [objective_function(val) for val in history_x], color='green', marker='o', s=10, label='Best x found per generation')
    plt.title('Objective Function and Best Solution History')
    plt.xlabel('x')
    plt.ylabel('f(x)')
    plt.grid(True)
    plt.legend()

    # Plot affinity over generations
    plt.subplot(1, 2, 2)
    plt.plot(generations, history_affinity, label='Overall Best Affinity', color='purple')
    plt.title('Overall Best Affinity Over Generations')
    plt.xlabel('Generation')
    plt.ylabel('Affinity (1 / (1 + f(x)))')
    plt.grid(True)
    plt.legend()

    plt.tight_layout()
    plt.show()