# -*- coding: utf-8 -*-
"""Dendritic_Cell_Algorithm-02.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/14HYMXjvpAG8WLMIC-5m4WGg5Qp7kZW1z
"""

import random
import numpy as np

class DendriticCell:
    """
    Represents a single Dendritic Cell (DC) in the simulation.
    It perceives signals, matures, and decides whether to migrate.
    """
    def __init__(self, cell_id, initial_location="tissue"):
        self.cell_id = cell_id
        self.location = initial_location # e.g., "tissue_area_1", "lymph_node"

        # Signal accumulators
        self.mcc = 0.0  # Mature Contextualized Cells (Danger-associated signals)
        self.psc = 0.0  # Pro-Survival Contextualized Cells (Safe/Homeostatic signals)
        self.csc = 0.0  # Co-Stimulatory Contextualized Cells (Inflammation-associated, important for T-cell activation)

        self.migratory_tendency = 0.0 # Propensity to migrate to lymph node
        self.status = "immature"     # immature, semi-mature, mature, migrating, exhausted

        self.signal_history = [] # To log received signals for analysis

    def receive_signals(self, danger_signal_strength, safe_signal_strength, csc_signal_strength):
        """
        Simulates the DC perceiving environmental signals.

        Args:
            danger_signal_strength (float): Magnitude of danger signals (e.g., PAMPs, DAMPs).
            safe_signal_strength (float): Magnitude of safe/homeostatic signals.
            csc_signal_strength (float): Magnitude of co-stimulatory signals (e.g., inflammatory cytokines).
        """
        self.mcc += danger_signal_strength
        self.psc += safe_signal_strength
        self.csc += csc_signal_strength

        # Store signals for history/debugging
        self.signal_history.append({
            'danger': danger_signal_strength,
            'safe': safe_signal_strength,
            'csc': csc_signal_strength
        })

        self._update_internal_state()

    def _update_internal_state(self):
        """
        Updates the DC's internal status and migratory tendency based on accumulated signals.
        This is a simplified maturation model.
        """

        # Calculate ratios for context
        # Avoid division by zero
        if (self.mcc + self.psc) == 0:
            danger_ratio = 0.0
            safe_ratio = 0.0
        else:
            danger_ratio = self.mcc / (self.mcc + self.psc)
            safe_ratio = self.psc / (self.mcc + self.psc)

        # Simplified maturation rules:
        # High danger and enough co-stimulation -> Mature
        if danger_ratio > 0.65 and self.csc > 3.0: # Thresholds are arbitrary for demonstration
            self.status = "mature"
            self.migratory_tendency = 1.0 # Mature DCs are highly migratory
        # High safe, or some danger but not enough co-stimulation -> Semi-mature or Immature
        elif safe_ratio > 0.6 or self.csc < 1.0:
            self.status = "immature" if self.mcc < 1.0 else "semi-mature"
            self.migratory_tendency = 0.1 # Immature/semi-mature are less migratory
        else:
            self.status = "semi-mature"
            self.migratory_tendency = 0.5 # Moderate tendency

        # Ensure migratory tendency is within bounds
        self.migratory_tendency = max(0.0, min(1.0, self.migratory_tendency))

        # Simulate exhaustion for very long-term high signals (optional)
        if self.mcc > 20.0 and self.psc < 5.0 and self.csc > 10.0:
            self.status = "exhausted"
            self.migratory_tendency = 0.0 # Exhausted cells stop responding/migrating

    def migrate(self):
        """
        Simulates the DC migrating to the 'lymph_node' if its migratory tendency is high
        or if it has achieved a 'mature' state.
        Returns True if migration occurred, False otherwise.
        """
        if self.location == "lymph_node" or self.status == "exhausted":
            return False # Already migrated or unable to migrate

        if self.status == "mature" or random.random() < self.migratory_tendency:
            print(f"  DC {self.cell_id} ({self.status}) migrating from {self.location}...")
            self.location = "lymph_node"
            self.status = "migrating" # Indicate it's in transit/arrived
            return True
        return False

    def get_context_for_t_cell(self):
        """
        Simulates the 'context' a DC would present to a T-cell in the lymph node.
        This is a critical output for anomaly detection in real DCA.
        Returns a tuple: (danger_context, safe_context, csc_context)
        """
        # A simple model: if MCC > PSC and CSC is high, it's a 'danger' context
        # Otherwise, if PSC is high, it's a 'safe' context

        # Normalize accumulated signals for context assessment (e.g., sum up)
        total_signals = self.mcc + self.psc + self.csc
        if total_signals == 0:
            return (0.0, 0.0, 0.0)

        danger_context = self.mcc / total_signals
        safe_context = self.psc / total_signals
        csc_context = self.csc / total_signals

        return (danger_context, safe_context, csc_context)

    def __str__(self):
        return (f"DC {self.cell_id:02d} | Loc: {self.location:<15} | Status: {self.status:<12} | "
                f"MCC: {self.mcc:6.2f}, PSC: {self.psc:6.2f}, CSC: {self.csc:6.2f} | "
                f"Mig.Tend: {self.migratory_tendency:.2f}")


class SimulationEnvironment:
    """
    Manages the overall DCA simulation, including DCs, signal generation,
    anomaly introduction, and result logging.
    """
    def __init__(self, num_dcs, tissue_areas, num_steps):
        self.num_dcs = num_dcs
        self.tissue_areas = tissue_areas
        self.num_steps = num_steps
        self.dendritic_cells = []
        self.simulation_log = []

        self._initialize_dcs()

    def _initialize_dcs(self):
        """Creates the initial set of Dendritic Cells."""
        print("--- Initializing Dendritic Cells ---")
        for i in range(self.num_dcs):
            initial_loc = random.choice(self.tissue_areas)
            dc = DendriticCell(i, initial_loc)
            self.dendritic_cells.append(dc)
            print(dc)
        print("-" * 50)

    def _generate_signals(self, current_step, dc_location):
        """
        Generates base signals for a DC based on the current step and location.
        This is where anomalies are injected.
        """
        # Default "normal" environment signals
        base_danger = random.uniform(0.1, 0.3)
        base_safe = random.uniform(0.8, 1.2)
        base_csc = random.uniform(0.1, 0.2) # Low co-stimulation in normal state

        danger = base_danger
        safe = base_safe
        csc = base_csc

        # --- ANOMALY INJECTION POINTS ---
        # Each anomaly targets specific conditions (step, location)

        # Anomaly 1: Acute Viral Infection (localized, strong, short-lived danger)
        if current_step >= 5 and current_step < 8 and dc_location == "tissue_area_A":
            danger = random.uniform(4.0, 7.0) # High PAMP/DAMP
            safe = random.uniform(0.05, 0.1) # Safe signals suppressed
            csc = random.uniform(2.0, 4.0) # High inflammatory cytokines
            print(f"  [ANOMALY 1] Acute Viral Infection in {dc_location}!")

        # Anomaly 2: Chronic Autoimmune Flare-up (persistent, moderate, mixed signals)
        elif current_step >= 10 and current_step < 15 and dc_location == "tissue_area_B":
            danger = random.uniform(1.0, 2.0) # Moderate DAMPs from tissue damage
            safe = random.uniform(0.5, 0.9)  # Some homeostatic signals still present
            csc = random.uniform(1.0, 2.5)   # Moderate ongoing inflammation
            print(f"  [ANOMALY 2] Chronic Autoimmune Flare in {dc_location}!")

        # Anomaly 3: Silent Tumor Growth (low danger, but increasing CSC over time)
        elif current_step >= 12 and dc_location == "tissue_area_C":
            danger = random.uniform(0.3, 0.7) # Low-level DAMPs (apoptosis, necrosis)
            safe = random.uniform(0.7, 1.0)   # Mostly normal otherwise
            # CSC increases with step, simulating tumor producing cytokines or causing chronic irritation
            csc = base_csc + (current_step - 10) * 0.2 # CSC slowly ramps up
            csc = min(csc, 5.0) # Cap CSC
            print(f"  [ANOMALY 3] Silent Tumor Growth in {dc_location}!")

        # Anomaly 4: Overwhelming Sepsis (systemic, very high danger, potential exhaustion)
        elif current_step == 18: # A single, very bad step
            danger = random.uniform(8.0, 12.0) # Massive PAMPs/DAMPs
            safe = random.uniform(0.01, 0.05) # Safe signals almost absent
            csc = random.uniform(5.0, 8.0) # Extreme inflammation
            print(f"  [ANOMALY 4] Systemic Sepsis! DCs will be overloaded!")

        return danger, safe, csc

    def run(self):
        """Executes the main simulation loop."""
        for step in range(self.num_steps):
            print(f"\n--- Simulation Step {step + 1}/{self.num_steps} ---")
            step_log = {'step': step + 1, 'dc_states': [], 'lymph_node_arrivals': []}

            for dc in self.dendritic_cells:
                if dc.location == "lymph_node" and dc.status != "exhausted":
                    # DCs in lymph node are presenting, not perceiving new tissue signals
                    # For simplicity, they might slowly clear their signals or remain active
                    # We'll just let them stay in lymph_node state and log their context
                    pass
                else:
                    danger, safe, csc = self._generate_signals(step, dc.location)
                    dc.receive_signals(danger, safe, csc)

                if dc.migrate():
                    step_log['lymph_node_arrivals'].append({
                        'cell_id': dc.cell_id,
                        'initial_tissue': dc.location, # Note: this is *before* migration, so it's the source tissue
                        'mcc': dc.mcc,
                        'psc': dc.psc,
                        'csc': dc.csc,
                        'context': dc.get_context_for_t_cell()
                    })

                step_log['dc_states'].append(str(dc))
                print(dc) # Print current state of each DC

            self.simulation_log.append(step_log)
            print("-" * 50)

            # Global Anomaly Detection System (based on lymph node arrivals)
            self._analyze_lymph_node_activity(step_log)

        print("\n--- Simulation Complete ---")
        self._summarize_results()

    def _analyze_lymph_node_activity(self, step_log):
        """
        Analyzes the activity in the 'lymph node' to detect systemic anomalies.
        This mimics how T-cells would get activated by mature DCs.
        """
        migrated_mature_dcs = [
            arrival for arrival in step_log['lymph_node_arrivals']
            if arrival['context'][0] > arrival['context'][1] # Danger > Safe
            and arrival['context'][2] > 0.3 # Sufficient CSC for activation
        ]

        if len(migrated_mature_dcs) >= 2: # Threshold for a significant alert
            print("\n!!! SYSTEM ALERT: Multiple 'danger-context' DCs arrived at lymph node this step! !!!")
            for dc_info in migrated_mature_dcs:
                print(f"  - DC {dc_info['cell_id']} from {dc_info['initial_tissue']} with context: "
                      f"Danger={dc_info['context'][0]:.2f}, Safe={dc_info['context'][1]:.2f}, CSC={dc_info['context'][2]:.2f}")
            print("!!! Potential Systemic Anomaly Detected !!!")
        elif len(migrated_mature_dcs) == 1:
             print("\n  (Warning: A single 'danger-context' DC arrived at lymph node this step.)")

    def _summarize_results(self):
        """Provides a summary of the simulation results."""
        print("\n--- Final Simulation Summary ---")
        final_lymph_node_dcs = [dc for dc in self.dendritic_cells if dc.location == "lymph_node"]

        if final_lymph_node_dcs:
            print(f"\n{len(final_lymph_node_dcs)} DCs migrated to the lymph node.")
            for dc in final_lymph_node_dcs:
                danger_c, safe_c, csc_c = dc.get_context_for_t_cell()
                print(f"- DC {dc.cell_id} (Final Status: {dc.status}): "
                      f"Final Context (D/S/C): {danger_c:.2f}/{safe_c:.2f}/{csc_c:.2f}")
                if danger_c > safe_c and csc_c > 0.3:
                    print("  -> This DC presented a 'DANGER' context to T-cells.")
                else:
                    print("  -> This DC presented a 'SAFE' or 'TOLEROGENIC' context to T-cells.")
        else:
            print("No DCs migrated to the lymph node during the simulation.")

        exhausted_dcs = [dc for dc in self.dendritic_cells if dc.status == "exhausted"]
        if exhausted_dcs:
            print(f"\n{len(exhausted_dcs)} DCs became exhausted.")
            for dc in exhausted_dcs:
                print(f"- DC {dc.cell_id} (MCC:{dc.mcc:.1f}, PSC:{dc.psc:.1f}, CSC:{dc.csc:.1f})")


def run_dca_full_simulation():
    """Main function to configure and start the DCA simulation."""
    random.seed(42) # For reproducible results
    np.random.seed(42)

    num_dcs = 8
    tissue_areas = ["tissue_area_A", "tissue_area_B", "tissue_area_C", "tissue_area_D"]
    num_steps = 20 # Number of simulation time steps

    env = SimulationEnvironment(num_dcs, tissue_areas, num_steps)
    env.run()

if __name__ == "__main__":
    run_dca_full_simulation()